import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getUsersCollection } from "@/lib/database/collections"
import bcrypt from "bcryptjs"

export async function GET(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const usersCollection = await getUsersCollection()
    const user = await usersCollection.findOne(
      { _id: new ObjectId(params.id) },
      { projection: { password: 0 } }, // Exclude password from response
    )

    if (!user) {
      return NextResponse.json({ success: false, error: "User not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, data: user })
  } catch (error) {
    console.error("Error fetching user:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch user" }, { status: 500 })
  }
}

export async function PUT(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const body = await request.json()

    const updateData: any = {
      name: body.name,
      email: body.email,
      role: body.role,
      phone: body.phone,
      position: body.position,
      salary: body.salary,
      branchId: body.branchId ? new ObjectId(body.branchId) : undefined,
      isActive: body.isActive,
      updatedAt: new Date(),
    }

    // Hash password if provided
    if (body.password) {
      updateData.password = await bcrypt.hash(body.password, 12)
    }

    const usersCollection = await getUsersCollection()
    const result = await usersCollection.updateOne({ _id: new ObjectId(params.id) }, { $set: updateData })

    if (result.matchedCount === 0) {
      return NextResponse.json({ success: false, error: "User not found" }, { status: 404 })
    }

    // Remove password from response
    delete updateData.password

    return NextResponse.json({ success: true, data: updateData })
  } catch (error) {
    console.error("Error updating user:", error)
    return NextResponse.json({ success: false, error: "Failed to update user" }, { status: 500 })
  }
}

export async function PATCH(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const body = await request.json()
    console.log("[v0] PATCH request body:", body)

    // Build update object with only provided fields
    const updateData: any = {
      updatedAt: new Date(),
    }

    if (body.branchId) {
      // Adding a single branch to existing branchIds array
      const usersCollection = await getUsersCollection()
      const existingUser = await usersCollection.findOne({ _id: new ObjectId(params.id) })

      if (existingUser) {
        const currentBranchIds = existingUser.branchIds || []
        const newBranchId = new ObjectId(body.branchId)

        // Check if branch is already assigned
        const branchAlreadyAssigned = currentBranchIds.some((id: ObjectId) => id.toString() === newBranchId.toString())

        if (!branchAlreadyAssigned) {
          updateData.branchIds = [...currentBranchIds, newBranchId]
        } else {
          return NextResponse.json({ success: false, error: "Branch already assigned to this user" }, { status: 400 })
        }
      } else {
        updateData.branchIds = [new ObjectId(body.branchId)]
      }
    }

    if (body.branchIds && Array.isArray(body.branchIds)) {
      // Setting multiple branches (replace existing)
      updateData.branchIds = body.branchIds.map((id: string) => new ObjectId(id))
    }

    // Handle other fields if provided
    if (body.name !== undefined) updateData.name = body.name
    if (body.email !== undefined) updateData.email = body.email
    if (body.role !== undefined) updateData.role = body.role
    if (body.phone !== undefined) updateData.phone = body.phone
    if (body.position !== undefined) updateData.position = body.position
    if (body.salary !== undefined) updateData.salary = body.salary
    if (body.isActive !== undefined) updateData.isActive = body.isActive

    // Hash password if provided
    if (body.password) {
      updateData.password = await bcrypt.hash(body.password, 12)
    }

    console.log("[v0] Update data:", updateData)

    const usersCollection = await getUsersCollection()
    const result = await usersCollection.updateOne({ _id: new ObjectId(params.id) }, { $set: updateData })

    console.log("[v0] Update result:", result)

    if (result.matchedCount === 0) {
      return NextResponse.json({ success: false, error: "User not found" }, { status: 404 })
    }

    // Remove password from response
    delete updateData.password

    return NextResponse.json({ success: true, data: updateData })
  } catch (error) {
    console.error("[v0] Error updating user:", error)
    return NextResponse.json({ success: false, error: "Failed to update user" }, { status: 500 })
  }
}

export async function DELETE(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const usersCollection = await getUsersCollection()

    // Soft delete by setting isActive to false instead of hard delete
    const result = await usersCollection.updateOne(
      { _id: new ObjectId(params.id) },
      {
        $set: {
          isActive: false,
          updatedAt: new Date(),
        },
      },
    )

    if (result.matchedCount === 0) {
      return NextResponse.json({ success: false, error: "User not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, message: "User deactivated successfully" })
  } catch (error) {
    console.error("Error deleting user:", error)
    return NextResponse.json({ success: false, error: "Failed to delete user" }, { status: 500 })
  }
}
